﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
// ReSharper disable ConvertNullableToShortForm

namespace LinqAsMonads
{
	[TestClass]
	public class _3_NullableWithLinq
	{
		[TestMethod]
		public void UseNullables()
		{
			Nullable<int> result = null;

			var input1 = TryReadNullableInt();
			if (input1.HasValue)
			{
				var input2 = TryReadNullableInt();
				if (input2.HasValue)
					result = input1.Value * input2.Value;
			}
		}

		public static Nullable<int> TryReadNullableInt()
		{
			var rnd = new Random();

			int next = rnd.Next(100);
			return next > 10 ? new Nullable<int>(next) : null;
		}

		[TestMethod]
		public void UseEnumerableAsNullable()
		{
			IEnumerable<int> result = new int[0];

			var input1 = TryReadEnumerableInt();
			if (input1.Any())
			{
				var input2 = TryReadEnumerableInt();
				if (input2.Any())
					result = new[] { input1.Single() * input2.Single() };
			}
		}

		static IEnumerable<int> TryReadEnumerableInt()
		{
			var rnd = new Random();

			int next = rnd.Next(100);
			return next > 10 ? new[] { next } : new int[0];
		}

		[TestMethod]
		public void UseEnumerableAsNullableForEach()
		{
			IEnumerable<int> result = new int[0];

			foreach (var input1 in TryReadEnumerableInt())
				foreach (var input2 in TryReadEnumerableInt())
					result = new[] { input1 * input2 };
		}

		[TestMethod]
		public void UseEnumerableAsNullableQCS()
		{
			IEnumerable<int> result = from input1 in TryReadEnumerableInt()
									  from input2 in TryReadEnumerableInt()

									  select input1 * input2;
		}

		[TestMethod]
		public void UseNullableWithQCS()
		{
			Nullable<int> result = from input1 in TryReadNullableInt()
								   from input2 in TryReadNullableInt()

								   select input1 * input2;

			var temp = result.Value;
		}

		[TestMethod]
		public void UseThreeLines()
		{
			MyNullable<int> result = from input1 in TryReadMyNullableInt()
									 from input2 in TryReadMyNullableInt()
									 from input3 in TryReadMyNullableInt()

									 select input1 * input2 * input3;

			var temp = result.Value;
		}

		public static MyNullable<int> TryReadMyNullableInt()
		{
			var rnd = new Random();

			int next = rnd.Next(100);
			return next > 10 ? new MyNullable<int>(next) : new MyNullable<int>();
		}

		// Not possible for standard .Net Nullable<T> since T must be a struct
		//If we use more than two from-clauses an anonymous type is used for T
		//to aggregate the bindings. The anonymous type is derived from "class".
		//[TestMethod]
		//public void UseNullables2()
		//{
		//    Nullable<int> result = from input1 in TryReadNullableInt()
		//                           from input2 in TryReadNullableInt()
		//                           from input3 in TryReadNullableInt()
		//                           select input1 * input2 * input3;

		//    var temp = result.Value;
		//}
	}

	static class NullableEx
	{
		public static Nullable<TResult> SelectMany<T, T2, TResult>(
			this Nullable<T> source,
			Func<T, Nullable<T2>> collectionSelector,
			Func<T, T2, TResult> resultSelector)
			where T : struct
			where T2 : struct
			where TResult : struct
		{
			if (source.HasValue)
			{
				var two = collectionSelector(source.Value);
				if (two.HasValue)
				{
					var result = resultSelector(source.Value, two.Value);
					return new Nullable<TResult>(result);
				}
			}
			return null;
		}
	}

	public class MyNullable<T>
	{
		public MyNullable() { }

		public MyNullable(T value)
		{
			Value = value;
			HasValue = true;
		}

		public T Value { get; private set; }
		public bool HasValue { get; private set; }

		public MyNullable<TResult> SelectMany<T2, TResult>(
			Func<T, MyNullable<T2>> collectionSelector,
			Func<T, T2, TResult> resultSelector)
		{
			if (HasValue)
			{
				var two = collectionSelector(Value);
				if (two.HasValue)
				{
					var result = resultSelector(Value, two.Value);
					return new MyNullable<TResult>(result);
				}
			}
			return new MyNullable<TResult>();
		}
	}
}
// ReSharper restore ConvertNullableToShortForm